/*********************************************************************
*	 Project: MG82F6P32-DEMO
*			TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV	Board
*			CpuCLK=12MHz, SysCLK=12MHz
*	 Description:
*			GPL	Bit	Order Reversed
*			GPL	CRC	calculate(CCIT16 CRC16 0x1021,CRC32:0x04C1_1DB7)
*			GPL	Flash automatic	CRC
*	 Note:
*
*	 Creat time:2024.02.26
*	 Modify:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*Set SysClk	(MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*Set CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)

#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26


code u8	TestBuf[16]={0xAA,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};


/*************************************************
*Function:		void DelayXus(u16 xUs)
*Description:		delay,unit:us
*Input:				u8 Us -> *1us  (1~255)
*Output:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:		void DelayXms(u16 xMs)
*Description:	 delay,unit:ms
*Input:				u16	xMs	-> *1ms	 (1~65535)
*Output:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*Function:		void SendByte(u8 tByte)
*Description:	send byte
*Input:			u8 tByte: the data to be send
*Output:
*************************************************************************************/
void SendByte(u8 tByte)
{
	bit	bES0;
	bES0=ES0;
	ES0=0;
	S0BUF=tByte;
	while(TI0==0);
	TI0=0;
	ES0=bES0;
}

/***********************************************************************************
*Function:		void SendStr(BYTE* PStr)
*Description:	send string
*Input:				u8*	PStr:the string	to be send
*Output:
*************************************************************************************/
void SendStr(u8* PStr)
{
	while(*PStr	!= 0)
	{
		SendByte(*PStr);
		PStr ++;
	}
}

/***********************************************************************************
*Function:		void SendHex(u8	ToSend)
*Description:	send data for HEX
*Input:				u8 ToSend: the data	to be send
*Output:
*************************************************************************************/
void SendHex(u8	ToSend)
{
	u8 x;
//	SendByte(' ');
//	SendByte('0');
//	SendByte('x');
	x =	ToSend >> 4;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
	x= ToSend&0x0F;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
}

/***********************************************************************************
*Function:		void SendDW2DEC(u32	ToSend)
*Description:	send data for DEC
*Input:				u32	ToSend:	the	data to	be send
*Output:
*************************************************************************************/
void SendDW2DEC(u32	ToSend)
{
	bit	bFlag;
	u8 x;
	bFlag=0;
	x=ToSend/1000000000L;
	ToSend=ToSend%1000000000L;
	if(x!=0)
	{	bFlag=TRUE;
		SendByte('0'+x);
	}

	x=ToSend/100000000L;
	ToSend=ToSend%100000000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}

	x=ToSend/10000000L;
	ToSend=ToSend%10000000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}

	x=ToSend/1000000L;
	ToSend=ToSend%1000000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}

	x=ToSend/100000L;
	ToSend=ToSend%100000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}

	x=ToSend/10000L;
	ToSend=ToSend%10000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}

	x=ToSend/1000L;
	ToSend=ToSend%1000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}

	x=ToSend/100L;
	ToSend=ToSend%100L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}

	x=ToSend/10L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}

	x=ToSend%10L;
	SendByte('0'+x);

}

/***********************************************************************************
*Function:		void ResetCounter()
*Description:	reset counter
*Input:
*Output:
*************************************************************************************/
void ResetCounter()
{
	// Stop	T0,T1
	TM_DisableT0();TM_DisableT1();
	TM_SetT0HighByte(0);
	TM_SetT0LowByte(0);
	TM_SetT1HighByte(0);
	TM_SetT1LowByte(0);
	// Start T0,T1
	TM_EnableT0();TM_EnableT1();
}
/***********************************************************************************
*Function:		u32	GetCounter()
*Description:	get	counter
*Input:
*Output:
*************************************************************************************/
u32	GetCounter()
{
	DWordTypeDef dwTime;
	TM_DisableT0();TM_DisableT1();
	dwTime.B.BHigh=TH0;
	dwTime.B.BMHigh=TL0;
	dwTime.B.BMLow=TH1;
	dwTime.B.BLow=TL1;
	return dwTime.DW;
}

/***********************************************************************************
*Function:		void InitTimer0(void)
*Description:	Initialize Timer0  Sysclk/192 6us~393216us
*Input:
*Output:
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// Set T0 mode:16bit timer
	TM_SetT0Clock_T1OF();					// Set T0 clock:T1OF
	TM_SetT0Gate_Disable();
	TM_SetT0LowByte(0);						// Set T0 low byte
	TM_SetT0HighByte(0);					// Set T0 high byte
	//TM_EnableT0();						// Enable T0
}

/***********************************************************************************
*Function:		void InitTimer1(void)
*Description:	Initialize Timer1  Sysclk/48 1.5us~98304us
*Input:
*Output:
*************************************************************************************/
void InitTimer1(void)
{
	TM_SetT1Mode_1_16BIT_TIMER();			// Set T1 mode:16bit timer
	TM_SetT1Clock_SYSCLKDiv12();			// Set T1 clock: SYSCLK/12
	TM_SetT1Gate_Disable();
	TM_SetT1LowByte(0);						// Set T1 low byte
	TM_SetT1HighByte(0);					// Set T1 high byte
	//TM_EnableT1();						// Enable T1
}

/***********************************************************************************
*Function:		void InitUart0_S0BRG()
*Description:	Initialize Uart0, The source of	baud rate was S0BRG
*Input:
*Output:
*************************************************************************************/
void InitUart0_S0BRG(void)
{
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS0BRG();			// B.R.	source:	S0BRG
	UART0_SetS0BRGBaudRateX2();			// S0BRG x2
	UART0_SetS0BRGSelSYSCLK();			// S0BRG clock source: SYSCLK
	UART0_SetRxTxP30P31();
	// Sets	B.R. value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_12000000_1T);

	UART0_EnS0BRG();					// Enable S0BRG
}

/***********************************************************************************
*Function:		void InitPort()
*Description:	Initialize IO Port
*Input:
*Output:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);				// Set P22,P24,P26 as push-pull(LED)
	PORT_SetP3QuasiBi(BIT0|BIT1);					// Set P30,P31 as Quasi-Bi
}

/***********************************************************************************
*Function:		void InitClock()
*Description:	Initialize clock
*Input:
*Output:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:		 void InitSystem(void)
*Description:	 Initialize	MCU
*Input:
*Output:
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitUart0_S0BRG();
	InitTimer0();
	InitTimer1();
}

/***********************************************************************************
*Function:		u16	CRC16_CCIT(u8 *ptr,u16 len)
*Description:	calculation	CRC16
*Input:
*				u8 *pBuf: the Buffer to	be CRC
*				u16	len: the length	of Buffer
*Output:
*				u16:CRC16
*************************************************************************************/
u16	CRC16_CCIT(u8 *ptr,u16 len)
{
	u16	wCRCin = 0x0000;
	u16	wCPoly = 0x1021;
	u16	wChar=0;
	u16	i;
	u8 x;
	for(i=0;i<len;i++)
	{
		wChar=(ptr[i])&0x00FF;
		wCRCin=wCRCin^(wChar<<8);
		for(x=0;x<8;x++)
		{
			if((wCRCin&0x8000)!=0)
			{
				wCRCin=wCRCin<<1;
				wCRCin=wCRCin^wCPoly;
			}
			else
			{
				wCRCin=wCRCin<<1;
			}
		}
	}
	return wCRCin;
}

/***********************************************************************************
*Function:		u16	GPL_CRC(u8 *pBuf,u8	len)
*Description:	calculation	CRC16 (0x1021)
*Input:
*				u8 *pBuf: the Buffer to	be CRC
*				u8 len:	the	length of Buffer
*Output:
*				u16:CRC16
*************************************************************************************/
u16	GPL_CRC(u8 *pBuf,u8	len)
{
	WordTypeDef	CRCValue;
	u8 i;
	// Set CRC Mode
	GPL_CRCMode_CRC16_0x1021();

	// Set CRC Seed
	GPL_CRC_WriteCRCSeedH(0x00);
	GPL_CRC_WriteCRCSeedL(0x00);

	GPL_CRC_CRC0DA_WriteDataIn();
	for(i=0;i<len;i++)
	{
		GPL_CRC_CRC0DA_Write(pBuf[i]);
	}
	GPL_CRC_ReadCRCResultH(CRCValue.B.BHigh);
	GPL_CRC_ReadCRCResultL(CRCValue.B.BLow);
	return CRCValue.W;
}

/***********************************************************************************
*Function:		u16	GPL_AutoFlashCRC(u16 StartAddr,u16 EndAddr)
*Description:	Automatic calculation CRC for flash
*Input:
*				u16	StartAddr: Flash start address
*				u16	EndAddr: Flash end address(	The	last byte of the flash page.
*							Ex:	....0x33FF,0x35FF,0x37FF,0x39FF,0x3BFF,0x3DFF,0x3FFF.)
*Output:
*				u16:CRC16
*************************************************************************************/
u16	GPL_AutoFlashCRC(u16 StartAddr,u16 EndAddr)
{
	WordTypeDef	CRCValue;
	bit	bEA=EA;					// Backup EA
	// Set CRC Mode
	GPL_CRCMode_CRC16_0x1021();

	// Set CRC Seed
	GPL_CRC_WriteCRCSeedH(0x00);
	GPL_CRC_WriteCRCSeedL(0x00);

	GPL_CRC_CRC0DA_WriteDataIn();

	// Set End address
	EA = 0;						// Disable interrupt
	IFADRH = 0;					// IFADRH must 0
	IFADRL=	IAPLB_P;			// IAPLB address
	IFD= (EndAddr>>8)&0xFE;		// IAPLB
	IFMT = ISP_WRITE_P;
	ISPCR =	0x80;				// start ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR =	0;					// clear ISPCR
	// Set Start address
	IFADRH = StartAddr/256;		//High address
	IFADRL=	StartAddr%256;		//Low address
	IFMT = ISP_AUTO_FLASH_CRC;
	ISPCR =	0x80;				// start ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	GPL_CRC_ReadCRCResultH(CRCValue.B.BHigh);
	GPL_CRC_ReadCRCResultL(CRCValue.B.BLow);
	IFMT=0;
	ISPCR =	0;					// clear ISPCR
	EA = bEA;					// restore EA
	return CRCValue.W;
}

/***********************************************************************************
*Function:	 u32 GPL_CRC32(u8 *pBuf,u8 len)
*Description: calculation CRC32:0x04C1_1DB7
*Input:
*				u8 *pBuf: the Buffer to	be CRC
*				u8 len:	the	length of Buffer
*Output:
*		 u32: CRC32
*************************************************************************************/
u32	GPL_CRC32(u8 *pBuf,u8 len)
{
	DWordTypeDef CRC32Value;
	u8 i;
	// Set CRC Mode
	GPL_CRCMode_CRC32_0x04C1_1DB7();

	// Set CRC Seed
	GPL_CRC_WriteCRC32Seed0(0xFF);
	GPL_CRC_WriteCRC32Seed1(0xFF);
	GPL_CRC_WriteCRC32Seed2(0xFF);
	GPL_CRC_WriteCRC32Seed3(0xFF);

	GPL_CRC_CRC0DA_WriteDataIn();
	for(i=0;i<len;i++)
	{
		GPL_CRC_CRC0DA_Write(pBuf[i]);
	}
	GPL_CRC_ReadCRC32Result0(CRC32Value.B.BLow);
	GPL_CRC_ReadCRC32Result1(CRC32Value.B.BMLow);
	GPL_CRC_ReadCRC32Result2(CRC32Value.B.BMHigh);
	GPL_CRC_ReadCRC32Result3(CRC32Value.B.BHigh);
	return CRC32Value.DW;
}

void main()
{
	WordTypeDef	CRCValue;
	u8 TestBorev;
	DWordTypeDef dwTime,CRC32Value;
	unsigned char volatile code	*pCode;

	InitSystem();

	SendStr("\nStart GPL Test!");

	IO_LED_G=0;IO_LED_Y=0;IO_LED_R=0;
	DelayXms(1000);
	IO_LED_G=1;IO_LED_Y=1;IO_LED_R=1;

	// Bit Order Reversed
	TestBorev=0x46;
	SendStr("\nBorev old:0x");
	SendHex(TestBorev);
	GPL_BOREV(TestBorev);
	SendStr("\nBorev new:0x");
	SendHex(TestBorev);


	// Calculate CRC16 of the FLASH	0x0000~0x3FFF
	ResetCounter();
	CRCValue.W=GPL_AutoFlashCRC(0x0000,	(1024-1));
	dwTime.DW=GetCounter();
	SendStr("\nHW 1KFlash CRC16	Value:0x");
	SendHex(CRCValue.B.BHigh);
	SendHex(CRCValue.B.BLow);
	SendStr("; Time:");
	SendDW2DEC(dwTime.DW);

	// Software	Calculate CRC16	of the FLASH 0x0000~0x3FFF
	pCode=0x0000;
	ResetCounter();
	CRCValue.W=CRC16_CCIT(pCode, 1024);
	dwTime.DW=GetCounter();
	SendStr("\nSW 1KFlash CRC16	Value:0x");
	SendHex(CRCValue.B.BHigh);
	SendHex(CRCValue.B.BLow);
	SendStr("; Time:");
	SendDW2DEC(dwTime.DW);

	// Calculate CRC16 of TestBuf[16]
	ResetCounter();
	CRCValue.W=GPL_CRC(TestBuf,16);
	dwTime.DW=GetCounter();
	SendStr("\nHW TestBuf[16] CRC16	Value:0x");
	SendHex(CRCValue.B.BHigh);
	SendHex(CRCValue.B.BLow);
	SendStr("; Time:");
	SendDW2DEC(dwTime.DW);

	// Software	Calculate CRC16	of TestBuf[16]
	ResetCounter();
	CRCValue.W=CRC16_CCIT(TestBuf,16);
	dwTime.DW=GetCounter();
	SendStr("\nSoft	TestBuf[16]	CRC16 Value:0x");
	SendHex(CRCValue.B.BHigh);
	SendHex(CRCValue.B.BLow);
	SendStr("; Time:");
	SendDW2DEC(dwTime.DW);


	// Calculate CRC32 of TestBuf[16]
	ResetCounter();
	CRC32Value.DW=GPL_CRC32(TestBuf,16);
	dwTime.DW=GetCounter();
	SendStr("\nHW TestBuf[16] CRC32	Value:0x");
	SendHex(CRC32Value.B.BHigh);
	SendHex(CRC32Value.B.BMHigh);
	SendHex(CRC32Value.B.BMLow);
	SendHex(CRC32Value.B.BLow);
	SendStr("; Time:");
	SendDW2DEC(dwTime.DW);
	while(1)
	{
		DelayXms(100);
		IO_LED_G = !IO_LED_G;
	}

}


